/*
 * Decompiled with CFR 0.152.
 */
package azkaban.scheduler;

import azkaban.scheduler.Schedule;
import azkaban.scheduler.ScheduleLoader;
import azkaban.scheduler.ScheduleManagerException;
import azkaban.utils.DataSourceUtils;
import azkaban.utils.GZIPUtils;
import azkaban.utils.JSONUtils;
import azkaban.utils.Props;
import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.sql.DataSource;
import org.apache.commons.dbutils.DbUtils;
import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.log4j.Logger;
import org.joda.time.DateTimeZone;
import org.joda.time.ReadablePeriod;

public class JdbcScheduleLoader
implements ScheduleLoader {
    private static Logger logger = Logger.getLogger(JdbcScheduleLoader.class);
    private DataSource dataSource;
    private EncodingType defaultEncodingType = EncodingType.GZIP;
    private static final String scheduleTableName = "schedules";
    private static String SELECT_ALL_SCHEDULES = "SELECT project_id, project_name, flow_name, status, first_sched_time, timezone, period, last_modify_time, next_exec_time, submit_time, submit_user, enc_type, schedule_options FROM schedules";
    private static String INSERT_SCHEDULE = "INSERT INTO schedules ( project_id, project_name, flow_name, status, first_sched_time, timezone, period, last_modify_time, next_exec_time, submit_time, submit_user, enc_type, schedule_options) values (?,?,?,?,?,?,?,?,?,?,?,?,?)";
    private static String REMOVE_SCHEDULE_BY_KEY = "DELETE FROM schedules WHERE project_id=? AND flow_name=?";
    private static String UPDATE_SCHEDULE_BY_KEY = "UPDATE schedules SET status=?, first_sched_time=?, timezone=?, period=?, last_modify_time=?, next_exec_time=?, submit_time=?, submit_user=?, enc_type=?, schedule_options=? WHERE project_id=? AND flow_name=?";
    private static String UPDATE_NEXT_EXEC_TIME = "UPDATE schedules SET next_exec_time=? WHERE project_id=? AND flow_name=?";

    private Connection getConnection() throws ScheduleManagerException {
        Connection connection = null;
        try {
            connection = this.dataSource.getConnection();
        }
        catch (Exception e) {
            DbUtils.closeQuietly((Connection)connection);
            throw new ScheduleManagerException("Error getting DB connection.", e);
        }
        return connection;
    }

    public EncodingType getDefaultEncodingType() {
        return this.defaultEncodingType;
    }

    public void setDefaultEncodingType(EncodingType defaultEncodingType) {
        this.defaultEncodingType = defaultEncodingType;
    }

    public JdbcScheduleLoader(Props props) {
        String databaseType = props.getString("database.type");
        if (databaseType.equals("mysql")) {
            int port = props.getInt("mysql.port");
            String host = props.getString("mysql.host");
            String database = props.getString("mysql.database");
            String user = props.getString("mysql.user");
            String password = props.getString("mysql.password");
            int numConnections = props.getInt("mysql.numconnections");
            this.dataSource = DataSourceUtils.getMySQLDataSource(host, port, database, user, password, numConnections);
        }
    }

    @Override
    public List<Schedule> loadSchedules() throws ScheduleManagerException {
        List schedules;
        logger.info((Object)"Loading all schedules from db.");
        Connection connection = this.getConnection();
        QueryRunner runner = new QueryRunner();
        ScheduleResultHandler handler = new ScheduleResultHandler();
        try {
            schedules = (List)runner.query(connection, SELECT_ALL_SCHEDULES, (ResultSetHandler)handler);
        }
        catch (SQLException e) {
            logger.error((Object)(SELECT_ALL_SCHEDULES + " failed."));
            throw new ScheduleManagerException("Loading schedules from db failed. ", e);
        }
        finally {
            DbUtils.closeQuietly((Connection)connection);
        }
        logger.info((Object)"Now trying to update the schedules");
        for (Schedule sched : schedules) {
            if (!sched.updateTime()) {
                logger.info((Object)("Schedule " + sched.getScheduleName() + " was scheduled before azkaban start, skipping it."));
                schedules.remove(sched);
                this.removeSchedule(sched);
                continue;
            }
            logger.info((Object)"Recurring schedule, need to update next exec time");
            try {
                this.updateNextExecTime(sched);
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new ScheduleManagerException("Update next execution time failed.", e);
            }
            logger.info((Object)("Schedule " + sched.getScheduleName() + " loaded and updated."));
        }
        logger.info((Object)("Loaded " + schedules.size() + " schedules."));
        return schedules;
    }

    @Override
    public void removeSchedule(Schedule s) throws ScheduleManagerException {
        logger.info((Object)("Removing schedule " + s.getScheduleName() + " from db."));
        QueryRunner runner = new QueryRunner(this.dataSource);
        try {
            int removes = runner.update(REMOVE_SCHEDULE_BY_KEY, new Object[]{s.getProjectId(), s.getFlowName()});
            if (removes == 0) {
                throw new ScheduleManagerException("No schedule has been removed.");
            }
        }
        catch (SQLException e) {
            logger.error((Object)(REMOVE_SCHEDULE_BY_KEY + " failed."));
            throw new ScheduleManagerException("Remove schedule " + s.getScheduleName() + " from db failed. ", e);
        }
    }

    @Override
    public void insertSchedule(Schedule s) throws ScheduleManagerException {
        logger.info((Object)("Inserting schedule " + s.getScheduleName() + " into db."));
        this.insertSchedule(s, this.defaultEncodingType);
    }

    public void insertSchedule(Schedule s, EncodingType encType) throws ScheduleManagerException {
        String json = JSONUtils.toJSON(s.optionsToObject());
        byte[] data = null;
        try {
            byte[] stringData;
            data = stringData = json.getBytes("UTF-8");
            if (encType == EncodingType.GZIP) {
                data = GZIPUtils.gzipBytes(stringData);
            }
            logger.debug((Object)("NumChars: " + json.length() + " UTF-8:" + stringData.length + " Gzip:" + data.length));
        }
        catch (IOException e) {
            throw new ScheduleManagerException("Error encoding the schedule options. " + s.getScheduleName());
        }
        QueryRunner runner = new QueryRunner(this.dataSource);
        try {
            int inserts = runner.update(INSERT_SCHEDULE, new Object[]{s.getProjectId(), s.getProjectName(), s.getFlowName(), s.getStatus(), s.getFirstSchedTime(), s.getTimezone().getID(), Schedule.createPeriodString(s.getPeriod()), s.getLastModifyTime(), s.getNextExecTime(), s.getSubmitTime(), s.getSubmitUser(), encType.getNumVal(), data});
            if (inserts == 0) {
                throw new ScheduleManagerException("No schedule has been inserted.");
            }
        }
        catch (SQLException e) {
            logger.error((Object)(INSERT_SCHEDULE + " failed."));
            throw new ScheduleManagerException("Insert schedule " + s.getScheduleName() + " into db failed. ", e);
        }
    }

    @Override
    public void updateNextExecTime(Schedule s) throws ScheduleManagerException {
        logger.info((Object)("Update schedule " + s.getScheduleName() + " into db. "));
        Connection connection = this.getConnection();
        QueryRunner runner = new QueryRunner();
        try {
            runner.update(connection, UPDATE_NEXT_EXEC_TIME, new Object[]{s.getNextExecTime(), s.getProjectId(), s.getFlowName()});
        }
        catch (SQLException e) {
            e.printStackTrace();
            logger.error((Object)(UPDATE_NEXT_EXEC_TIME + " failed."), (Throwable)e);
            throw new ScheduleManagerException("Update schedule " + s.getScheduleName() + " into db failed. ", e);
        }
        finally {
            DbUtils.closeQuietly((Connection)connection);
        }
    }

    @Override
    public void updateSchedule(Schedule s) throws ScheduleManagerException {
        logger.info((Object)("Updating schedule " + s.getScheduleName() + " into db."));
        this.updateSchedule(s, this.defaultEncodingType);
    }

    public void updateSchedule(Schedule s, EncodingType encType) throws ScheduleManagerException {
        String json = JSONUtils.toJSON(s.optionsToObject());
        byte[] data = null;
        try {
            byte[] stringData;
            data = stringData = json.getBytes("UTF-8");
            if (encType == EncodingType.GZIP) {
                data = GZIPUtils.gzipBytes(stringData);
            }
            logger.debug((Object)("NumChars: " + json.length() + " UTF-8:" + stringData.length + " Gzip:" + data.length));
        }
        catch (IOException e) {
            throw new ScheduleManagerException("Error encoding the schedule options " + s.getScheduleName());
        }
        QueryRunner runner = new QueryRunner(this.dataSource);
        try {
            int updates = runner.update(UPDATE_SCHEDULE_BY_KEY, new Object[]{s.getStatus(), s.getFirstSchedTime(), s.getTimezone().getID(), Schedule.createPeriodString(s.getPeriod()), s.getLastModifyTime(), s.getNextExecTime(), s.getSubmitTime(), s.getSubmitUser(), encType.getNumVal(), data, s.getProjectId(), s.getFlowName()});
            if (updates == 0) {
                throw new ScheduleManagerException("No schedule has been updated.");
            }
        }
        catch (SQLException e) {
            logger.error((Object)(UPDATE_SCHEDULE_BY_KEY + " failed."));
            throw new ScheduleManagerException("Update schedule " + s.getScheduleName() + " into db failed. ", e);
        }
    }

    public class ScheduleResultHandler
    implements ResultSetHandler<List<Schedule>> {
        public List<Schedule> handle(ResultSet rs) throws SQLException {
            if (!rs.next()) {
                return Collections.emptyList();
            }
            ArrayList<Schedule> schedules = new ArrayList<Schedule>();
            do {
                int projectId = rs.getInt(1);
                String projectName = rs.getString(2);
                String flowName = rs.getString(3);
                String status = rs.getString(4);
                long firstSchedTime = rs.getLong(5);
                DateTimeZone timezone = DateTimeZone.forID((String)rs.getString(6));
                ReadablePeriod period = Schedule.parsePeriodString(rs.getString(7));
                long lastModifyTime = rs.getLong(8);
                long nextExecTime = rs.getLong(9);
                long submitTime = rs.getLong(10);
                String submitUser = rs.getString(11);
                int encodingType = rs.getInt(12);
                byte[] data = rs.getBytes(13);
                Object optsObj = null;
                if (data != null) {
                    EncodingType encType = EncodingType.fromInteger(encodingType);
                    try {
                        String jsonString;
                        if (encType == EncodingType.GZIP) {
                            jsonString = GZIPUtils.unGzipString(data, "UTF-8");
                            optsObj = JSONUtils.parseJSONFromString(jsonString);
                        } else {
                            jsonString = new String(data, "UTF-8");
                            optsObj = JSONUtils.parseJSONFromString(jsonString);
                        }
                    }
                    catch (IOException e) {
                        throw new SQLException("Error reconstructing schedule options " + projectName + "." + flowName);
                    }
                }
                Schedule s = new Schedule(projectId, projectName, flowName, status, firstSchedTime, timezone, period, lastModifyTime, nextExecTime, submitTime, submitUser);
                if (optsObj != null) {
                    s.createAndSetScheduleOptions(optsObj);
                }
                schedules.add(s);
            } while (rs.next());
            return schedules;
        }
    }

    public static enum EncodingType {
        PLAIN(1),
        GZIP(2);

        private int numVal;

        private EncodingType(int numVal) {
            this.numVal = numVal;
        }

        public int getNumVal() {
            return this.numVal;
        }

        public static EncodingType fromInteger(int x) {
            switch (x) {
                case 1: {
                    return PLAIN;
                }
                case 2: {
                    return GZIP;
                }
            }
            return PLAIN;
        }
    }
}

