/*
 * Decompiled with CFR 0.152.
 */
package azkaban.project;

import azkaban.flow.Flow;
import azkaban.project.Project;
import azkaban.project.ProjectLoader;
import azkaban.project.ProjectLogEvent;
import azkaban.project.ProjectManagerException;
import azkaban.user.Permission;
import azkaban.user.User;
import azkaban.utils.DirectoryFlowLoader;
import azkaban.utils.Props;
import azkaban.utils.Utils;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.zip.ZipFile;
import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;

public class ProjectManager {
    private static final Logger logger = Logger.getLogger(ProjectManager.class);
    private ConcurrentHashMap<Integer, Project> projectsById = new ConcurrentHashMap();
    private ConcurrentHashMap<String, Project> projectsByName = new ConcurrentHashMap();
    private final ProjectLoader projectLoader;
    private final Props props;
    private final File tempDir;
    private final int projectVersionRetention;
    private final boolean creatorDefaultPermissions;

    public ProjectManager(ProjectLoader loader, Props props) {
        this.projectLoader = loader;
        this.props = props;
        this.tempDir = new File(this.props.getString("project.temp.dir", "temp"));
        this.projectVersionRetention = props.getInt("project.version.retention", 3);
        logger.info((Object)("Project version retention is set to " + this.projectVersionRetention));
        this.creatorDefaultPermissions = props.getBoolean("creator.default.proxy", true);
        if (!this.tempDir.exists()) {
            this.tempDir.mkdirs();
        }
        this.loadAllProjects();
    }

    private void loadAllProjects() {
        List projects;
        try {
            projects = this.projectLoader.fetchAllActiveProjects();
        }
        catch (ProjectManagerException e) {
            throw new RuntimeException("Could not load projects from store.", e);
        }
        for (Project proj : projects) {
            this.projectsByName.put(proj.getName(), proj);
            this.projectsById.put(proj.getId(), proj);
        }
        for (Project proj : projects) {
            this.loadAllProjectFlows(proj);
        }
    }

    private void loadAllProjectFlows(Project project) {
        try {
            List flows = this.projectLoader.fetchAllProjectFlows(project);
            HashMap<String, Flow> flowMap = new HashMap<String, Flow>();
            for (Flow flow : flows) {
                flowMap.put(flow.getId(), flow);
            }
            project.setFlows(flowMap);
        }
        catch (ProjectManagerException e) {
            throw new RuntimeException("Could not load projects flows from store.", e);
        }
    }

    public List<String> getProjectNames() {
        return new ArrayList<String>(this.projectsByName.keySet());
    }

    public List<Project> getUserProjects(User user) {
        ArrayList<Project> array = new ArrayList<Project>();
        for (Project project : this.projectsById.values()) {
            Permission perm = project.getUserPermission(user);
            if (perm == null || !perm.isPermissionSet(Permission.Type.ADMIN) && !perm.isPermissionSet(Permission.Type.READ)) continue;
            array.add(project);
        }
        return array;
    }

    public List<Project> getGroupProjects(User user) {
        ArrayList<Project> array = new ArrayList<Project>();
        for (Project project : this.projectsById.values()) {
            if (!project.hasGroupPermission(user, Permission.Type.READ)) continue;
            array.add(project);
        }
        return array;
    }

    public List<Project> getUserProjectsByRegex(User user, String regexPattern) {
        Pattern pattern;
        ArrayList<Project> array = new ArrayList<Project>();
        try {
            pattern = Pattern.compile(regexPattern, 2);
        }
        catch (PatternSyntaxException e) {
            logger.error((Object)("Bad regex pattern " + regexPattern));
            return array;
        }
        for (Project project : this.projectsById.values()) {
            Permission perm = project.getUserPermission(user);
            if (perm == null || !perm.isPermissionSet(Permission.Type.ADMIN) && !perm.isPermissionSet(Permission.Type.READ) || !pattern.matcher(project.getName()).find()) continue;
            array.add(project);
        }
        return array;
    }

    public List<Project> getProjects() {
        return new ArrayList<Project>(this.projectsById.values());
    }

    public List<Project> getProjectsByRegex(String regexPattern) {
        Pattern pattern;
        ArrayList<Project> allProjects = new ArrayList<Project>();
        try {
            pattern = Pattern.compile(regexPattern, 2);
        }
        catch (PatternSyntaxException e) {
            logger.error((Object)("Bad regex pattern " + regexPattern));
            return allProjects;
        }
        for (Project project : this.getProjects()) {
            if (!pattern.matcher(project.getName()).find()) continue;
            allProjects.add(project);
        }
        return allProjects;
    }

    public Project getProject(String name) {
        return (Project)this.projectsByName.get(name);
    }

    public Project getProject(int id) {
        return (Project)this.projectsById.get(id);
    }

    public Project createProject(String projectName, String description, User creator) throws ProjectManagerException {
        if (projectName == null || projectName.trim().isEmpty()) {
            throw new ProjectManagerException("Project name cannot be empty.");
        }
        if (description == null || description.trim().isEmpty()) {
            throw new ProjectManagerException("Description cannot be empty.");
        }
        if (creator == null) {
            throw new ProjectManagerException("Valid creator user must be set.");
        }
        if (!projectName.matches("[a-zA-Z][a-zA-Z_0-9|-]*")) {
            throw new ProjectManagerException("Project names must start with a letter, followed by any number of letters, digits, '-' or '_'.");
        }
        if (this.projectsByName.contains(projectName)) {
            throw new ProjectManagerException("Project already exists.");
        }
        logger.info((Object)("Trying to create " + projectName + " by user " + creator.getUserId()));
        Project newProject = this.projectLoader.createNewProject(projectName, description, creator);
        this.projectsByName.put(newProject.getName(), newProject);
        this.projectsById.put(newProject.getId(), newProject);
        if (this.creatorDefaultPermissions) {
            this.projectLoader.updatePermission(newProject, creator.getUserId(), new Permission(new Permission.Type[]{Permission.Type.ADMIN}), false);
            newProject.addProxyUser(creator.getUserId());
            try {
                this.updateProjectSetting(newProject);
            }
            catch (ProjectManagerException e) {
                e.printStackTrace();
                throw e;
            }
        }
        this.projectLoader.postEvent(newProject, ProjectLogEvent.EventType.CREATED, creator.getUserId(), null);
        return newProject;
    }

    public synchronized Project removeProject(Project project, User deleter) throws ProjectManagerException {
        this.projectLoader.removeProject(project, deleter.getUserId());
        this.projectLoader.postEvent(project, ProjectLogEvent.EventType.DELETED, deleter.getUserId(), null);
        this.projectsByName.remove(project.getName());
        this.projectsById.remove(project.getId());
        return project;
    }

    public void updateProjectDescription(Project project, String description, User modifier) throws ProjectManagerException {
        this.projectLoader.updateDescription(project, description, modifier.getUserId());
        this.projectLoader.postEvent(project, ProjectLogEvent.EventType.DESCRIPTION, modifier.getUserId(), "Description changed to " + description);
    }

    public List<ProjectLogEvent> getProjectEventLogs(Project project, int results, int skip) throws ProjectManagerException {
        return this.projectLoader.getProjectEvents(project, results, skip);
    }

    public Props getProperties(Project project, String source) throws ProjectManagerException {
        return this.projectLoader.fetchProjectProperty(project, source);
    }

    public Props getJobOverrideProperty(Project project, String jobName) throws ProjectManagerException {
        return this.projectLoader.fetchProjectProperty(project, String.valueOf(jobName) + ".jor");
    }

    public void setJobOverrideProperty(Project project, Props prop, String jobName) throws ProjectManagerException {
        prop.setSource(String.valueOf(jobName) + ".jor");
        Props oldProps = this.projectLoader.fetchProjectProperty(project, prop.getSource());
        if (oldProps == null) {
            this.projectLoader.uploadProjectProperty(project, prop);
        } else {
            this.projectLoader.updateProjectProperty(project, prop);
        }
    }

    public void updateProjectSetting(Project project) throws ProjectManagerException {
        this.projectLoader.updateProjectSettings(project);
    }

    public void addProjectProxyUser(Project project, String proxyName, User modifier) throws ProjectManagerException {
        logger.info((Object)("User " + modifier.getUserId() + " adding proxy user " + proxyName + " to project " + project.getName()));
        project.addProxyUser(proxyName);
        this.projectLoader.postEvent(project, ProjectLogEvent.EventType.PROXY_USER, modifier.getUserId(), "Proxy user " + proxyName + " is added to project.");
        this.updateProjectSetting(project);
    }

    public void removeProjectProxyUser(Project project, String proxyName, User modifier) throws ProjectManagerException {
        logger.info((Object)("User " + modifier.getUserId() + " removing proxy user " + proxyName + " from project " + project.getName()));
        project.removeProxyUser(proxyName);
        this.projectLoader.postEvent(project, ProjectLogEvent.EventType.PROXY_USER, modifier.getUserId(), "Proxy user " + proxyName + " has been removed form the project.");
        this.updateProjectSetting(project);
    }

    public void updateProjectPermission(Project project, String name, Permission perm, boolean group, User modifier) throws ProjectManagerException {
        logger.info((Object)("User " + modifier.getUserId() + " updating permissions for project " + project.getName() + " for " + name + " " + perm.toString()));
        this.projectLoader.updatePermission(project, name, perm, group);
        if (group) {
            this.projectLoader.postEvent(project, ProjectLogEvent.EventType.GROUP_PERMISSION, modifier.getUserId(), "Permission for group " + name + " set to " + perm.toString());
        } else {
            this.projectLoader.postEvent(project, ProjectLogEvent.EventType.USER_PERMISSION, modifier.getUserId(), "Permission for user " + name + " set to " + perm.toString());
        }
    }

    public void removeProjectPermission(Project project, String name, boolean group, User modifier) throws ProjectManagerException {
        logger.info((Object)("User " + modifier.getUserId() + " removing permissions for project " + project.getName() + " for " + name));
        this.projectLoader.removePermission(project, name, group);
        if (group) {
            this.projectLoader.postEvent(project, ProjectLogEvent.EventType.GROUP_PERMISSION, modifier.getUserId(), "Permission for group " + name + " removed.");
        } else {
            this.projectLoader.postEvent(project, ProjectLogEvent.EventType.USER_PERMISSION, modifier.getUserId(), "Permission for user " + name + " removed.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void uploadProject(Project project, File archive, String fileType, User uploader) throws ProjectManagerException {
        File file;
        block14: {
            logger.info((Object)("Uploading files to " + project.getName()));
            file = null;
            try {
                if (fileType == null) {
                    throw new ProjectManagerException("Unknown file type for " + archive.getName());
                }
                if ("zip".equals(fileType)) {
                    file = this.unzipFile(archive);
                    break block14;
                }
                throw new ProjectManagerException("Unsupported archive type for file " + archive.getName());
            }
            catch (IOException e) {
                throw new ProjectManagerException("Error unzipping file.", (Throwable)e);
            }
        }
        logger.info((Object)("Validating Flow for upload " + archive.getName()));
        DirectoryFlowLoader loader = new DirectoryFlowLoader(logger);
        loader.loadProjectFlow(file);
        if (!loader.getErrors().isEmpty()) {
            logger.error((Object)("Error found in upload to " + project.getName() + ". Cleaning up."));
            try {
                FileUtils.deleteDirectory((File)file);
            }
            catch (IOException e) {
                file.deleteOnExit();
                e.printStackTrace();
            }
            StringBuffer errorMessage = new StringBuffer();
            errorMessage.append("Error found in upload. Cannot upload.\n");
            for (String error : loader.getErrors()) {
                errorMessage.append(error);
                errorMessage.append('\n');
            }
            throw new ProjectManagerException(errorMessage.toString());
        }
        Map jobProps = loader.getJobProps();
        List propProps = loader.getProps();
        Project project2 = project;
        synchronized (project2) {
            int newVersion = this.projectLoader.getLatestProjectVersion(project) + 1;
            Map flows = loader.getFlowMap();
            for (Flow flow : flows.values()) {
                flow.setProjectId(project.getId());
                flow.setVersion(newVersion);
            }
            logger.info((Object)("Uploading file to db " + archive.getName()));
            this.projectLoader.uploadProjectFile(project, newVersion, fileType, archive.getName(), archive, uploader.getUserId());
            logger.info((Object)("Uploading flow to db " + archive.getName()));
            this.projectLoader.uploadFlows(project, newVersion, flows.values());
            logger.info((Object)("Changing project versions " + archive.getName()));
            this.projectLoader.changeProjectVersion(project, newVersion, uploader.getUserId());
            project.setFlows(flows);
            logger.info((Object)"Uploading Job properties");
            this.projectLoader.uploadProjectProperties(project, new ArrayList(jobProps.values()));
            logger.info((Object)"Uploading Props properties");
            this.projectLoader.uploadProjectProperties(project, propProps);
        }
        logger.info((Object)"Uploaded project files. Cleaning up temp files.");
        this.projectLoader.postEvent(project, ProjectLogEvent.EventType.UPLOADED, uploader.getUserId(), "Uploaded project files zip " + archive.getName());
        try {
            FileUtils.deleteDirectory((File)file);
        }
        catch (IOException e) {
            file.deleteOnExit();
            e.printStackTrace();
        }
        logger.info((Object)("Cleaning up old install files older than " + (project.getVersion() - this.projectVersionRetention)));
        this.projectLoader.cleanOlderProjectVersion(project.getId(), project.getVersion() - this.projectVersionRetention);
    }

    public void updateFlow(Project project, Flow flow) throws ProjectManagerException {
        this.projectLoader.updateFlow(project, flow.getVersion(), flow);
    }

    private File unzipFile(File archiveFile) throws IOException {
        ZipFile zipfile = new ZipFile(archiveFile);
        File unzipped = Utils.createTempDir((File)this.tempDir);
        Utils.unzip((ZipFile)zipfile, (File)unzipped);
        return unzipped;
    }

    public void postProjectEvent(Project project, ProjectLogEvent.EventType type, String user, String message) {
        this.projectLoader.postEvent(project, type, user, message);
    }
}

